/**
* \file: message_sender.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "app_iface/message_sender.h"

#include "utils/logger.h"
#include "utils/automounter_types_internal.h"
#include "ipc/request_messages.h"
#include "control/automounter.h"

//------------------------------------- public member definition ------------------------------------------------------
void message_sender_send_request_done_msg(app_iface_request_ctx_t *request_ctx, error_code_t result,
		const char *error_msg)
{
	message_buffer_t *msg_buffer=socket_server_get_empty_msg_buffer();
	logger_log_debug("Sending request (id=%d) done message to application '%s':",
			request_ctx->request_id, socket_server_get_connection_appidentifier(request_ctx->ctx));
	logger_log_debug("		Error code: %d, Message: %s", result, error_msg);
	if (request_message_prepare_request_done_msg(msg_buffer,result,error_msg,
			request_ctx->request_id, request_ctx->remote_callback_ptr)!=RESULT_OK)
		automounter_die_on_resource_issues();

	//socket_server_send_msg will put back the message buffer when it is not needed anymore
	//we are not interested in the result, if the sending fails, the connected is removed automaticallyy within the
	//send function
	(void)socket_server_send_msg(msg_buffer,request_ctx->ctx);
}

void message_sender_send_partition_event_msg(info_event_t event,
		const device_info_t *dev_info, const partition_info_t *part_info)
{
	message_buffer_t *msg_buffer=socket_server_get_empty_msg_buffer();

	logger_log_debug("Broadcasting partition info message to the applications:");
	logger_log_debug("		Partition: %s", part_info->interface_id);
	if (info_messages_prepare_part_info_msg(msg_buffer, event, dev_info,part_info,-1)!=RESULT_OK)
		automounter_die_on_resource_issues();

	//socket_server_broadcast_msg will put back the message buffer when it is not needed anymore
	socket_server_broadcast_msg(msg_buffer,(event_mask_t)event);
}

void message_sender_send_device_event_msg(info_event_t event, const device_info_t *dev_info)
{
	message_buffer_t *msg_buffer=socket_server_get_empty_msg_buffer();

	logger_log_debug("Broadcasting device info message to the applications:");
	logger_log_debug("		Device: %s", dev_info->interface_id);
	if (info_messages_prepare_dev_info_msg(msg_buffer,event, dev_info,-1)!=RESULT_OK)
		automounter_die_on_resource_issues();

	//socket_server_broadcast_msg will put back the message buffer when it is not needed anymore
	socket_server_broadcast_msg(msg_buffer,(event_mask_t)event);
}

error_code_t message_sender_send_snapshot_done_msg(connection_ctx_t *ctx, int request_id)
{
	message_buffer_t *msg_buffer=socket_server_get_empty_msg_buffer();

	logger_log_debug("Sending snapshot request done message (request_id: %d) to application '%s'.",
			request_id, socket_server_get_connection_appidentifier(ctx));
	if (request_message_prepare_snapshot_done_msg(msg_buffer,request_id)!=RESULT_OK)
		automounter_die_on_resource_issues();

	//socket_server_send_msg will put back the message buffer when it is not needed anymore
	return socket_server_send_msg(msg_buffer,ctx);
}

error_code_t message_sender_send_device_update_msg(connection_ctx_t *ctx, const device_info_t *dev_info,int request_id)
{
	message_buffer_t *msg_buffer;

	//application didn't register a callback for this event. So don't send it.
	if (!socket_server_is_ctx_registered_for_event(ctx, (event_mask_t)DEVICE_UPDATE_INFO_EVENT)) return RESULT_OK;

	msg_buffer=socket_server_get_empty_msg_buffer();
	logger_log_debug("Going to send an info update message for device %s (request_id=%d) to application '%s'.",
			dev_info->interface_id, request_id, socket_server_get_connection_appidentifier(ctx));
	if (info_messages_prepare_dev_info_msg(msg_buffer,DEVICE_UPDATE_INFO_EVENT, dev_info,request_id)!=RESULT_OK)
		automounter_die_on_resource_issues();

	//socket_server_send_msg will put back the message buffer when it is not needed anymore
	return socket_server_send_msg(msg_buffer,ctx);
}

error_code_t message_sender_send_partition_update_msg(connection_ctx_t *ctx, const device_info_t *dev_info,
		const partition_info_t *part_info, int request_id)
{
	message_buffer_t *msg_buffer;

	//application didn't register a callback for this event. So don't send it.
	if (!socket_server_is_ctx_registered_for_event(ctx, (event_mask_t)PARTITION_UPDATE_INFO_EVENT))	return RESULT_OK;

	msg_buffer=socket_server_get_empty_msg_buffer();
	logger_log_debug("Going to send an info update message for partition %s (request_id=%d) to application '%s'.",
			part_info->interface_id,request_id,socket_server_get_connection_appidentifier(ctx));
	if (info_messages_prepare_part_info_msg(msg_buffer,PARTITION_UPDATE_INFO_EVENT, dev_info,part_info,request_id)!=RESULT_OK)
			automounter_die_on_resource_issues();

	//socket_server_send_msg will put back the message buffer when it is not needed anymore
	return socket_server_send_msg(msg_buffer,ctx);
}
//---------------------------------------------------------------------------------------------------------------------
